################################################################################
##              Maschinelles Lernen-Projekt Kreuzvalidierung
##          Konzeptstudie zur Ausführung auf einem Raspberry Pi5
## 
## Motivation:
## Das Konzept der Kreuzvalidierung wird angewendet, wenn die Datenmenge klein
## ist (< 1000 Datenpunkte?).
## Allgemein kann gesagt werden, dass bei einer kleinen Datenmenge Über-
## anpassung drohen kann.
## Das Netzes kann also bei einem einmaligen Training eine große Varianz
## beinhalten, verursacht durch die geringe Größe der Trainings-
## und Validierungsdatensätze.
## Zur Ermittlung der optimalen Trainingsparameter kann das Aufteilen der Daten 
## in k-Mengen und das k-fache Training des indentischen Modells hilfreich sein.
## Da es sich in diesem Beispiel um ein Regressionsmodell handelt, dient der
## der Güteparameter "mittlerer absolute Fehler" (mae) zur Optimierung der Trainings-
## parameter.
## 
##
##
## Ad-Oculos-Projekt, https://www.faes.de/ad-oculos/
## Günter Faes
## Version 0.0.2, 08.05.2025
## R-Version: 4.4.2  
## Pi5 OS: Debian GNU/Linux 12 (bookworm) # Über cat /etc/os-release
##
################################################################################
## Pakete:
##  Zur Installation und möglichen Problemen, siehe Skript Pi5_NN_MNIST.R
##  YouTube: https://youtu.be/rEClRl2-t_U

# Startzeit Optimierung über Kreuzvalidierung:
Skript_Startzeit <- format(Sys.time(),"%d-%m-%Y %H:%M:%S") 
cat("\n","Starte Skript um ", Skript_Startzeit, "\n", "\n")


library(reticulate)
# Bevor irgendetwas mit Python ausgeführt wird, wird die notwendige Conda-Umgebung
# eingestellt:
reticulate::use_condaenv("/home/collector1/miniconda3/envs/my_env")

library(keras)
library(lubridate, warn.conflicts = FALSE) # Zur Zeitformatierung

############################## Beispieldaten ##################################

# Der Boston-Housing-Price-Datensatz wird über keras zur Verfügung gestellt und
# heruntergeladen:

#Der Datensatz enthält in jedem Fall 14 Attribute. Diese sind:
# X:
#   CRIM - Pro-Kopf-Kriminalitätsrate pro Stadt
#   ZN - Anteil der Wohnbauflächen, die für Grundstücke mit einer Größe von mehr als 25.000 Quadratmetern ausgewiesen sind
#   INDUS - Anteil der Flächen für Nicht-Einzelhandelsunternehmen pro Stadt.
#   CHAS - Charles River Dummy-Variable (1, wenn der Trakt an den Fluss grenzt; 0 andernfalls)
#   NOX - Stickoxid-Konzentration (Teile pro 10 Millionen)
#   RM - durchschnittliche Anzahl der Zimmer pro Wohnung
#   AGE - Anteil der vor 1940 gebauten Eigentumswohnungen
#   DIS - gewichtete Entfernungen zu fünf Bostoner Beschäftigungszentren
#   RAD - Index der Erreichbarkeit radialer Autobahnen
#   TAX - Vollwert-Grundsteuerrate pro 10.000 $
#   PTRATIO - Schüler-Lehrer-Verhältnis pro Stadt
#   B - 1000(Bk - 0. 63)^2, wobei Bk der Anteil der Schwarzen in der Stadt ist
#   LSTAT - % niedrigerer Status der Bevölkerung
#
# Y:
#   MEDV - Medianwert von Eigenheimen in 1000 $


HausDaten <- dataset_boston_housing()

# Erstellen des Tranings- und Testdatensazes:
Train_x <- HausDaten$train$x; # View(Train_x)
Train_y <- HausDaten$train$y; # View(Train_y)
Test_x  <- HausDaten$test$x;  # View(Test_x)
Test_y  <- HausDaten$test$y;  # View(Test_y)

# z-Normierung der Daten durchführen:
MW <- apply(Train_x, 2, mean)
Stdabw <- apply(Train_x, 2, sd)
Train_x <- scale(Train_x, center = MW, scale = Stdabw)
Test_x <- scale(Test_x, center = MW, scale = Stdabw)

########################## NN-Modell erzeugen ####################################
# Da im Konzept der Kreuzvalidierung das gleiche Modell mehrfach genutzt wird,
# wird es als Funktion angelegt:

erzeuge_Modell <- function() {
  modell <- keras_model_sequential() %>%
    layer_dense(units = 64, activation = "relu", input_shape = dim(Train_x)[[2]]) %>%
    layer_dense(units = 64, activation = "relu") %>%
    layer_dense(units = 1)   # ohne Aktivierungsfunktion, um den Bereich nicht einzuschränken
  
  modell %>% compile(
    optimizer = "rmsprop",
    loss = "mse",        # Mean Squared Error
    metrics = c("mae")   # Mean Absolute Error
  )
  
}
###################### Optimale Trainingsparameter ermitteln #####################

## Variablen:
nk <- 4                                  # Anzahl Kreuzvalidierungen
ZufallsIndex <- sample(1:nrow(Train_x))  # Zufällige Indizes ziehen
# k Teilmengen bilden:
Teilmengen <- split(ZufallsIndex, cut(ZufallsIndex, breaks = nk, lables = FALSE))
n_Epochen <- 500                        # Anzahl Trainingsepochen
k_mae_history <- NULL
Ergebnis_Validierungsdaten_mae_Werte <- NULL


## Optimierung:
# (Nur auf Basis der Trainingsdaten!)
cat("Anzahl Teilmengen k: ", nk, "\n")

# Startzeit Optimierungs-Training:
startzeit <- now()

for (i in 1 : nk) {
  
  cat("Teilmenge: ", i, "\n")
  
  # Aktuelle Teilmenge:
  Validierung_Index <- Teilmengen[[i]]
  Training_Index <- unlist(Teilmengen[-i])
  
  # Aktuelle Trainingsdaten:
  TM_Training_x <- Train_x[Training_Index,]
  TM_Training_y <- Train_y[Training_Index]
  
  # Aktuelle Validierungsdaten:
  TM_Validierung_x <- Train_x[Validierung_Index,]
  TM_Validierung_y <- Train_y[Validierung_Index]
  
  # Modell bilden:
  Modell <- erzeuge_Modell()
  
  # Modell trainieren:
  history <- Modell %>% fit(TM_Training_x, TM_Training_y,
                            epochs = n_Epochen,
                            batch_size = 1,
                            verbose = 0)
  
  # Trainingsverlauf dokumentieren:
  mae_history <- history$metrics$mae
  k_mae_history <- rbind(k_mae_history, mae_history)
  
  # verbose = 0: Stille "Information", verbose = 2: ein paar Informationen
  Ergebnis_Validierungsdaten <- Modell %>% evaluate(TM_Validierung_x, TM_Validierung_y, verbose = 2)
  
  Ergebnis_Validierungsdaten_mae_Werte <- c(Ergebnis_Validierungsdaten_mae_Werte, Ergebnis_Validierungsdaten[2])
  
}  # Ende Optimierung 

# Ende Optimierungs-Training:
endezeit <- now()

### Grafische Ausgabe mae als PDF-Datei:
MW_mae_history <- data.frame(
  Epoche = seq(1 : ncol(k_mae_history)),
  Validierung_mae = apply(k_mae_history, 2, mean)
)

pdf("MAE_k_Kreuzvalidierung.pdf")
plot(Validierung_mae ~ Epoche, data = MW_mae_history,
     main = c(paste("Mittlerer absolute Fehler im Verlauf der", nk, "-fachen-Validierung")),
     xlab = "Epochen",
     ylab = "Mittlerer absolute Fehler (mae)")
dev.off()

## Infoausgabe Konsole:
print(paste("Start der k-Kreuzvalidierung: ", startzeit))
print(paste("Ende der k-Kreuzvalidierung: ", endezeit))
Optimierungsdauer <- as.duration(endezeit - startzeit)
print(paste("Optimierungsdauer: ", Optimierungsdauer))
cat("\n", "Optimierungs-Training, k-Validierungs-mae-Werte: ", Ergebnis_Validierungsdaten_mae_Werte, "\n")
cat("Optimierungs-Training, Mittelwert k-Validierungs-mae-Werte: ", mean(Ergebnis_Validierungsdaten_mae_Werte), "\n", "\n")


#### Skript-Ende #####
