#############################################################
# Ad-Olulos-Projekt: IoT-Simulation mit dem RPi Pico
#
#               Temperatursensor DS18B20
# 
# Projekt 4:
# In diesem Projekt werden die Daten von 2 Temperatursensoren
# DS18B20 über die One-Wire-Schnittstelle eingelesen und vorverarbeitet.
# Die Temperaturmessung wird auf Basis von 2 Kriterien überwacht:
#
#        1. Die Überschreitung eines Schwellwertes (ESW) und
#        2. ob eine untere bzw. obere Warngrenze überschritten wird.
#
# Die Ausgabe der Messwerte findet über ein OLED-Display statt und
# die Ereignisse werden über 2 RGB-Dioden signalisiert.
#
# Die Messwerte werden über MQTT publiziert und nachfolgend mit
# einem R-Skript auf einem Raspberry Pi ausgewertet und in eine
# Datenbank zur weiteren Auswertung gespeichert.
#
# Günter Faes
# Ad-Oculos-Projekt: https://www.faes.de/ad-oculos/
# MicroPython-Version: 1.21.0
# Version: 0.1.4  22.03.2024
#############################################################

# Libraries importieren:
import utime                            # Zeit
import machine                          # Hardware, GPIO
import ssd1306                          # OLED-Display
import ds3231                           # RTC (Uhr)
import network                          # WLAN
import ustruct as struct                # Byte -> dec umwandeln

from onewire import OneWire             # OneWire: serieller 1-Draht-Bus
from ds18x20 import DS18X20             # Temperatursensor DSS18B20
from umqtt.simple import MQTTClient     # Datenübertragung mit MQTT
                                        # Doku zu umqtt.simple: https://mpython.readthedocs.io/en/master/library/mPython/umqtt.simple.html



## GPIO-Pin-Deklaration:
# OneWire, Temperatursensoren DSS18B20
OW_Bus = machine.Pin(2) # GP2
# GP-Pin 14: Ein = Führe Hauptproramm aus:
Button_PIN14 = machine.Pin(14, machine.Pin.IN, machine.Pin.PULL_DOWN)
# IC2_1 für die RTC festlegen:
rtc = ds3231.RTC(sda_pin = 26, scl_pin = 27, port = 1)
# Pin-Festlegung für RGB-LED Sensor 1:
LED_S1_Blau  = machine.Pin(3, machine.Pin.OUT)
LED_S1_Gruen = machine.Pin(4, machine.Pin.OUT)
LED_S1_Rot   = machine.Pin(5, machine.Pin.OUT)
# Pin-Festlegung für RGB-LED Sensor 2:
LED_S2_Blau  = machine.Pin(6, machine.Pin.OUT)
LED_S2_Gruen = machine.Pin(7, machine.Pin.OUT)
LED_S2_Rot   = machine.Pin(8, machine.Pin.OUT)


## Festlegung der OLED-Display(ssd1306)-Eigenschaften über I2C, GP0 & GP1:
# Pin -> I2C:
sda0 = machine.Pin(0)
scl1 = machine.Pin(1)
i2c0 = machine.I2C(0, sda = sda0, scl = scl1, freq = 400000)
# Eigenschaften
OLED_width = 128
OLED_height = 64
OLED = ssd1306.SSD1306_I2C(OLED_width, OLED_height, i2c0)


## Sensor initalisieren
TempSensor = DS18X20(OneWire(OW_Bus))
# OneWire-Sensor-Adresse ermitteln:
SensorAdr = TempSensor.scan()

# -------------------- Funktionen -----------------------------
# Ausgabe auf dem OLED-Display:
def OLED_Ausgabe(text1 = "",
                  text2 = "",
                  text3 = "",
                  text4 = "",
                  text5 = "",
                  text6 = ""):
    
    # Display-Ausgabe:
    OLED.fill(0) # Display löschen
    OLED.show()
                
    OLED.text(text1, 0, 0, 1)
    OLED.text(text2, 0, 10, 1)
    OLED.text(text3, 0, 20, 1)
    OLED.text(text4, 0, 30, 1)
    OLED.text(text5, 0, 40, 1)
    OLED.text(text6, 0, 50, 1)
    
    
    OLED.show()

# -------------------------------------------------------------
# Temperatursensor auslesen:

def readT(Sensor):
    
    TempSensor.convert_temp()                     # Temperatur in °C
    Temperatur = TempSensor.read_temp(Sensor)     # Temperatur auslesen
    
    return Temperatur


# -------------------------------------------------------------
# WLAN-Aktivierung:
def wlanConnect():
    
    network.country("DE")                   # Ländereinstellung
    ssid = "... Ihr WLAN ..."
    password = "... Ihr WLAN-Passwort ..."
    wlan = network.WLAN(network.STA_IF)     # Client-Betrieb
    wlan.active(True)
    wlan.config(pm = 0xa11140)              # WLAN-Powersave wird ausgeschaltet
    wlan.connect(ssid, password)            # WLAN-Verbindung hestellen

    # WLAN-Verbindungsstatus prüfen:
    # WLAN-Status prüfen:
    OLED_Ausgabe(text1 = "Warten auf",
                 text2 = "WLAN-Verb.!")
    
    while not wlan.isconnected() and wlan.status() >= 0:
        utime.sleep(1)

    OLED_Ausgabe(text1 = "WLAN-Verb.",
                 text2 = "hergestellt!",
                 text3 = "Status: " + str(wlan.status()))
    
    utime.sleep(2)
    
# -------------------------------------------------------------
# WLAN deaktivieren:

def wlanDisconnect():
    
    network.country("DE")                   # Ländereinstellung
    wlan = network.WLAN(network.STA_IF)     # Client-Betrieb

    while wlan.isconnected():
        wlan.disconnect()
        

    OLED_Ausgabe(text1 = "WLAN-Verb.",
                 text2 = "unterbrochen!",
                 text3 = "Status: " + str(wlan.status()))
 
    utime.sleep(2)

# -------------------------------------------------------------
# Publishing-Zeichenkette erstellen:

def MakePublishingString(Datum = "",
                         SensorIndex = 0,
                         SensorName = "",
                         Zyklus = 0,
                         Messtemperatur = 0,
                         ESW = 0,
                         WarngrenzeUnten = 0,
                         WarngrenzeOben = 0,
                         EreignisFlag = ""):
    
    PString = Datum
    PString = PString + "," + str(SensorIndex)
    PString = PString + "," + SensorName
    PString = PString + "," + str(Zyklus)
    PString = PString + "," + str(Messtemperatur)
    PString = PString + "," + str(ESW)
    PString = PString + "," + str(WarngrenzeUnten)
    PString = PString + "," + str(WarngrenzeOben)
    PString = PString + "," + EreignisFlag
    
    print(PString)
    
    return PString

# -------------------------------------------------------------
# Verbinden mit dem MQTT-Server:

def connectMQTT():
    client = MQTTClient(client_id = b"Projekt_4",
                        server = b"e1.....086.s2.eu.hivemq.cloud",
                        port = 8883,
                        user = b"AdOculosProjekt",
                        password = b"... HiveMQ-Passwort ...",
                        keepalive = 7200,
                        ssl = True,
                        ssl_params = {"server_hostname": "e1.....086.s2.eu.hivemq.cloud"}
                        )
    
    client.connect()
    
    return client

# -------------------------------------------------------------
# Daten über MQTT publizieren:

def publizieren(thema, wert):
    
    # print("MQTT-Topic: ", thema, ", Nachricht: ", wert)
    
    client.publish(thema, wert)
    
    # print("Publiziert!")



# ------------------ Ausführungsteil: -------------------------

## Variablendeklarierung:
# Es wird von 2 Temperatursensoren ausgegangen!
SensorZaehler = 0
Sensorbezeichnung = ["Objekt A", "Objekt B"]
n_Sensoren = 0            # Teil der Sensorprüfrpoutine!
Temperatur_C = 0
MesszyklusZeit = 5        # Sekunden
MesswertZeit = ""         # Zeitstempel, der publiziert wird
messWertString = ""       # Publizier-Zeichenkette


# Pre-Datenanalyse:
ESW_S1 = 1                # Ereignisschwellwert, 1 °C
ESW_S2 = 1                # Ereignisschwellwert, 1 °C
Warngrenze_S1_unten = 15  # Warngrenze °C Sensor 1 unten
Warngrenze_S1_oben = 30   # Warngernze °C Sensor 1 oben
Warngrenze_S2_unten = 20  # Warngrenze °C Sensor 1 unten
Warngrenze_S2_oben = 40   # Warngernze °C Sensor 1 oben
EreignisFlag_S1 = ""      # Ereignis: Ja, kein Ereignis: Nein
EreignisFlag_S2 = ""

pre_T_S1 = 0   # Letzte Temperatur vor der erneuten Temperaturmessung
pre_T_S2 = 0

# Publisieren:
PublishingString_S1 = ""    # Enthält die Informationen des 1. Sensors
PublishingString_S2 = ""    #    "              "        "  2. Sensors
PublishingString_S1S2 = ""  # Informationen Sensor 1 & 2 zur Veröffentlichung

# Allgemeines:
wlanOn = False          # WLAN-Status

#---------------------------------------------------------------------------
# Ausführungsschleife:
while True:
    
    # Führe Messroutine solange aus, bis Button_PIN14 == 0:         
    if Button_PIN14.value() == 1:
               
        # WLAN einschalten:
        if wlanOn == False:
            wlanConnect()
            wlanOn = True
            
            # Mit MQTT-Broker verbinden:
            client = connectMQTT()
         
                
        # Beim Start des Messzykluses eine Sensorprüfung durchführen:
        if pre_T_S1 == 0:  
            n_Sensoren = len(SensorAdr)    # Anzahl der ermittelten Sensoren
            OLED_Ausgabe(text1 = "Initalisierung..",
                         text2 = "Sensoren: " + str(n_Sensoren))
            
            utime.sleep(1)

        # RGBs auf Messung setzen (grün):
        LED_S1_Gruen.value(1)
        LED_S2_Gruen.value(1)

        # Temperatur messen:
        for Sensor in SensorAdr:
            
            # RTC auslesen:
            rtc_Datum = rtc.ReadTime("ISO-8601")     # Zur Dokumentation
            rtc_Zeit = rtc.ReadTime("time")          # Für OLED_Ausgabe

            Temperatur_C = readT(Sensor)
            
            Sensor_num = struct.unpack(">hh", Sensor)  # Sensoradressen in num wandeln
                                    
            # Displayausgabe:
            if SensorZaehler == 0:    # Text für den 1. Sensor
                
                ausgabeText1 = "Zeit: " + rtc_Zeit
                ausgabeText2 = "T in Grad C"
                ausgabeText3 = "Zyklus: " + str(MesszyklusZeit) + " Sek."
                ausgabeText4 = Sensorbezeichnung[SensorZaehler] + ": " + str(Temperatur_C)
                
                # 1. Ausgabe auf dem OLED-Display:
                OLED_Ausgabe(text1 = ausgabeText1,
                             text2 = ausgabeText2,
                             text3 = ausgabeText3,
                             text4 = ausgabeText4)


                # ESW prüfen:
                # Im ersten Messzyklus ist pre_T_S1 = 0, deswegen mit dem aktuellen T-Wert belegen:
                if pre_T_S1 == 0:
                    pre_T_S1 = Temperatur_C
                
                # Temperatur steigt über ESW oder Warngrenze_S1_oben:
                if (Temperatur_C >= (pre_T_S1 + ESW_S1)) or (Temperatur_C >= Warngrenze_S1_oben):
                    
                    LED_S1_Gruen(0)  # RGB grün aus
                    LED_S1_Rot(1)    # RGB rot ein
                    EreignisFlag_S1 = "Ja"
                    MesszyklusZeit = 2   # Messzykluszeit auf 2 Sekunden gesetzt
                
                # Temperatur sinkt über ESW oder Warngrenze_S1_unten:
                elif (Temperatur_C <= (pre_T_S1 - ESW_S1)) or (Temperatur_C <= Warngrenze_S1_unten):
                    
                    LED_S1_Gruen(0)
                    LED_S1_Blau(1)
                    EreignisFlag_S1 = "Ja"
                    MesszyklusZeit = 2   # Messzykluszeit auf 2 Sekunden gesetzt
                    
                else:
                    
                    LED_S1_Rot(0)
                    LED_S1_Blau(0)
                    LED_S1_Gruen(1)
                    EreignisFlag_S1 = "Nein"
                    
                    # Messzykluszeit nur bei Flag-Gleichheit "Nein" anpassen:
                    if EreignisFlag_S1 == "Nein" and EreignisFlag_S2 == "Nein":
                        MesszyklusZeit = 5   # Messzykluszeit auf 5 Sekunden zurückgesetzt

                
                MesswertZeit = rtc_Datum + " " + rtc_Zeit
                PublishingString_S1 = MakePublishingString(MesswertZeit,Sensor_num[SensorZaehler],Sensorbezeichnung[SensorZaehler],MesszyklusZeit,Temperatur_C,ESW_S1,Warngrenze_S1_unten,Warngrenze_S1_oben,EreignisFlag_S1)

                # Messwert zur nächsten Prüfung speichern:
                pre_T_S1 = Temperatur_C
                

            elif SensorZaehler == 1:    # Text für den 2. Sensor
                
                ausgabeText5 = Sensorbezeichnung[SensorZaehler] + ": " + str(Temperatur_C)

                # Im ersten Messzyklus ist pre_T_S2 = 0, deswegen mit dem aktuellen T-Wert belegen:
                if pre_T_S2 == 0:
                    pre_T_S2 = Temperatur_C
                    
                # Temperatur steigt über ESW oder Warngrenze_S2_oben:
                if (Temperatur_C >= (pre_T_S2 + ESW_S2)) or (Temperatur_C >= Warngrenze_S2_oben):
                    
                    LED_S2_Gruen(0)  # RGB grün aus
                    LED_S2_Rot(1)    # RGB rot ein
                    EreignisFlag_S2 = "Ja"
                    MesszyklusZeit = 2   # Messzykluszeit auf 2 Sekunden gesetzt
                    
                # Temperatur sinkt über ESW oder Warngrenze_S2_unten:    
                elif (Temperatur_C <= (pre_T_S2 - ESW_S2)) or (Temperatur_C <= Warngrenze_S2_unten):
                    
                    LED_S2_Gruen(0)
                    LED_S2_Blau(1)
                    EreignisFlag_S2 = "Ja"
                    MesszyklusZeit = 2   # Messzykluszeit auf 2 Sekunden gesetzt
                    
                else:
                    
                    LED_S2_Rot(0)
                    LED_S2_Blau(0)
                    LED_S2_Gruen(1)
                    EreignisFlag_S2 = "Nein"
                    
                    # Messzykluszeit nur bei Flag-Gleichheit "Nein" anpassen:
                    if EreignisFlag_S1 == "Nein" and EreignisFlag_S2 == "Nein":
                        MesszyklusZeit = 5   # Messzykluszeit auf 5 Sekunden zurückgesetzt


                # Messwert zur nächsten Prüfung speichern:
                pre_T_S2 = Temperatur_C

                # Nun erfolgt die Textausgabe auf dem OLED-Display:
                OLED_Ausgabe(text1 = ausgabeText1,
                             text2 = ausgabeText2,
                             text3 = "Zyklus: " + str(MesszyklusZeit) + " Sek.",
                             text4 = ausgabeText4,
                             text5 = ausgabeText5)
                
                PublishingString_S2 = MakePublishingString(MesswertZeit,Sensor_num[SensorZaehler],Sensorbezeichnung[SensorZaehler],MesszyklusZeit,Temperatur_C,ESW_S2,Warngrenze_S2_unten,Warngrenze_S2_oben,EreignisFlag_S2)
                
                #Es wird nur eine Zeichenkette mit allen Messwerten publiziert:
                PublishingString_S1S2 = PublishingString_S1 + "," + PublishingString_S2 + "\n"
                # Übergeben an MQTT-Broker:
                publizieren("Messwert", PublishingString_S1S2)

            
            
            SensorZaehler += 1
        
        SensorZaehler = 0        # Sensorzähler für den nächsten for-Durchlauf zurücksetzen
        
        # print() # Kosolenausgabe zur Prüfung
        
        utime.sleep(MesszyklusZeit)
      
      
    else:
        
        # Führe Standby-Code solange aus, bis Button_PIN14 == 0:
        # (Standby)
        
        MesszyklusZeit = 1
        utime.sleep(MesszyklusZeit)
        
        # WLAN ausschalten:
        if wlanOn == True:
            wlanDisconnect()
            wlanOn = False

            # MQTT-Verbindung schließen:
            client.disconnect()

        # Display-Ausgabe:
        OLED_Ausgabe(text1 = "Standby")
        
        # Alle RGBs auf Standby setzen (ausschalten):
        LED_S1_Rot(0)
        LED_S1_Blau(0)
        LED_S1_Gruen(0)
        
        LED_S2_Rot(0)
        LED_S2_Blau(0)
        LED_S2_Gruen(0)
        
        # Vergleichstemperaturen auf 0 setzen:
        pre_T_S1 = 0
        pre_T_S2 = 0

       
#--------------------------Ende-------------------------------------------- 
        
     


