#############################################################
# Ad-Olulos-Projekt: IoT-Simulation mit dem RPi Pico
#
#           Sensor DHT22 (Feuchte & Temperatur)
# 
# Projekt 2, Datenübertragung mit MQTT:
# Die Messergebnisse Temperatur und Luftfeuchtigkeit werden
# mit Zeitstempel und sind mittels MQTT über das Internet von
# einem PC auslesbar.
# Das Auslesen und Verarbeiten der Daten wird von einem R-Skript
# durchgeführt.
#
# Die Basis für dieses MicroPython-Programm ist Sensor_DHT22_P1.py.
# Erweiterungen zu _P1.py werden mit P2 gekennzeichnet und alle
# File-Aktivitäten wurden gelöscht.
#
# Günter Faes
# Ad-Oculos-Projekt: https://www.faes.de/ad-oculos/
# MicroPython-Version: 1.20.0
# Version: 0.3.7  15.08.2023
#############################################################

# Libraries importieren:
import utime                            # Zeit
import machine                          # Hardware, GPIO
import ssd1306                          # OLED-Display
from dht import DHT22                   # Temperatur- / Luftfeuchtesensor DHT22
import ds3231                           # RTC (Uhr)
import network                          # P2, WLAN
from umqtt.simple import MQTTClient     # P2, Datenübertragung mit MQTT
                                        # Doku zu umqtt.simple: https://mpython.readthedocs.io/en/master/library/mPython/umqtt.simple.html


## GPIO-Pin-Deklaration:
# GP-Pin 14: Ein = Führe Hauptproramm aus:
Button_PIN14 = machine.Pin(14, machine.Pin.IN, machine.Pin.PULL_DOWN)
# GP-Pin 15: Grüne Led zeigt Hauptprogrammausführung an:
LED_PIN15 = machine.Pin(15, machine.Pin.OUT)
# P2, GP-Pin 13: Blaue LED zeigt an, dass das WLAN aktiv ist:
LED_PIN13 = machine.Pin(13, machine.Pin.OUT)
# GP-Pin 11: Eingangsignal des DHT22-Sensors:
dht22_sensor = DHT22(machine.Pin(11, machine.Pin.IN, machine.Pin.PULL_UP))
# IC2_1 für die RTC festlegen:
rtc = ds3231.RTC(sda_pin = 26, scl_pin = 27, port = 1)

# Festlegung der OLED-Display(ssd1306)-Eigenschaften über I2C, GP0 & GP1:
sda0 = machine.Pin(0)
scl1 = machine.Pin(1)
i2c0 = machine.I2C(0, sda = sda0, scl = scl1, freq = 400000)
# 
OLED_width = 128
OLED_height = 64
OLED = ssd1306.SSD1306_I2C(OLED_width, OLED_height, i2c0)

## --------------------- Funktionsbereich ----------------------

# Ausgabe auf dem OLED-Display:
def OLED_Ausgabe(text1 = "",
                  text2 = "",
                  text3 = "",
                  text4 = "",
                  text5 = "",
                  text6 = ""):
    
    # Display-Ausgabe:
    OLED.fill(0) # Display löschen
    OLED.show()
                
    OLED.text(text1, 0, 0, 1)
    OLED.text(text2, 0, 10, 1)
    OLED.text(text3, 0, 20, 1)
    OLED.text(text4, 0, 30, 1)
    OLED.text(text5, 0, 40, 1)
    OLED.text(text6, 0, 50, 1)
    
    
    OLED.show()

# -------------------------------------------------------------
# Temperatur- und Luftfeuchtigkeitsmessung mit DHT22 durchführen:

def SensorDHT22():
    
    dht22_sensor.measure()    # Messung durchführen
    # Werte lesen
    temp = dht22_sensor.temperature()
    humi = dht22_sensor.humidity()
    
    return temp, humi

# -------------------------------------------------------------
# P2 WLAN-Aktivierung:
def wlanConnect():
    
    network.country("DE")                   # Ländereinstellung
    ssid = "Gast-WLAN-xxxx"
    password = "Ihr Passwort"
    wlan = network.WLAN(network.STA_IF)     # Client-Betrieb
    wlan.active(True)
    wlan.config(pm = 0xa11140)              # WLAN-Powersave wird ausgeschaltet
    wlan.connect(ssid, password)            # WLAN-Verbindung hestellen

    # WLAN-Verbindungsstatus prüfen:
    # WLAN-Status prüfen:
    OLED_Ausgabe(text1 = "Warten auf",
                 text2 = "WLAN-Verb.!")
    
    while not wlan.isconnected() and wlan.status() >= 0:
        utime.sleep(1)

    OLED_Ausgabe(text1 = "WLAN-Verb.",
                 text2 = "hergestellt!",
                 text3 = "Status: " + str(wlan.status()))
    
    utime.sleep(2)
    
# -------------------------------------------------------------
# P2 WLAN deaktivieren:

def wlanDisconnect():
    
    network.country("DE")                   # Ländereinstellung
    wlan = network.WLAN(network.STA_IF)     # Client-Betrieb

    while wlan.isconnected():
        wlan.disconnect()
        

    OLED_Ausgabe(text1 = "WLAN-Verb.",
                 text2 = "unterbrochen!",
                 text3 = "Status: " + str(wlan.status()))
 
    utime.sleep(2)


# -------------------------------------------------------------
# P2 Verbinden mit dem MQTT-Server:

def connectMQTT():
    client = MQTTClient(client_id = b"DHT22_P2",
                        server = b"e16f8f38d9074f9380eca47887cb6086.s2.eu.hivemq.cloud",
                        port = 8883,
                        user = b"AdOculosProjekt",
                        password = b"Ihr HiveMQ-Passwort",
                        keepalive = 7200,
                        ssl = True,
                        ssl_params = {"server_hostname": "e16f8f38d9074f9380eca47887cb6086.s2.eu.hivemq.cloud"}
                        )
    
    client.connect()
    
    return client

# -------------------------------------------------------------
# P2 Messwerte über MQTT publizieren:

def publizieren(thema, wert):
    
    # print("MQTT-Topic: ", thema, ", Nachricht: ", wert)
    
    client.publish(thema, wert)
    
    # print("Publiziert!")



# ------------------ Ausführungsteil: -------------------------

wlanOn = False          # P2, Ist das Wlan eingeschaltet?
MesszyklusZeit = 0      # Inital-Messzyklus-Zeit, gilt nur beim Einschalten des Messvorganges

# Die Ausführungsschleife:
while True:
             
    #### Führe Messroutine solange aus, bis Button_PIN14 == 0:         
    if Button_PIN14.value() == 1:
        
        # P2, Nur mit erstem Durchlauf des Messzyklus durchführen:
        if wlanOn == False:
            
            # WLAN einschalten:
            wlanConnect()
            
            wlanOn = True
            LED_PIN13.value(1) # WLAN-LED einschalten
            
            # Mit MQTT-Broker verbinden:
            client = connectMQTT()
        
        # LED einschalten:
        LED_PIN15.value(1)
                                           
        # Lufttemperatur & Feuchtigkeit messen
        utime.sleep(MesszyklusZeit)
        F_Erg = SensorDHT22()
        Temperatur = str(F_Erg[0])
        Feuchtigkeit = str(F_Erg[1])
        
        # RTC auslesen:
        rtc_DatumZeit = rtc.ReadTime("DIN-1355-1+time") # Für Dateiausgabe
        rtc_Zeit = rtc.ReadTime("time")                 # Für OLED_Ausgabe
            
        OLED_Ausgabe(text1 = "Zeit: " + rtc_Zeit,
                     text2 = "T Grad C: " + Temperatur,
                     text3 = "Feuchtigk. %:",
                     text4 = Feuchtigkeit)
            
        # Veröffentlichungszeichenkette erstellen (MQTT):
        # Parametertrennung durch Leerzeichen
        messwertString = rtc_DatumZeit + "," + Temperatur + "," + Feuchtigkeit + "\n"
        
        # Übergeben an MQTT-Broker:
        publizieren("Messwert", messwertString)
        
        
        # Messzykluszeit auf Betriebszeit setzen, z. B. 3 Sekunden:
        MesszyklusZeit = 3
 
        
    else:
        
        #### Führe Standby-Code solange aus, bis Button_PIN14 == 0:
        # Messroutine-LED ausschalten:
        LED_PIN15.value(0)
        
        # P2, WLAN ausschalten udn MQTT-Verbindung schließen:
        if wlanOn == True:
            
            wlanDisconnect()
            wlanOn = False
            LED_PIN13.value(0)
            
            # MQTT-Verbindung schließen:
            client.disconnect()
        
        
        MesszyklusZeit = 1
        utime.sleep(MesszyklusZeit)
        
        # Display-Ausgabe:
        OLED_Ausgabe(text1 = "Standby")




