################################################################################
##                      Raspberry Pi Pico-IoT-Projekte
##
##                    Projekt DHT22-Sensordatenauswertung
## 
## IoT-Projekt 2:
## Der Sensor DHT22 (AM2302) ist ein Luftfeuchte- und Temperatursensor und in 
## diesem R-Skript werden die über das MicroPython-Programm Sensor_DHT22_P2.py
## gesammelten Daten beispielhaft ausgewertet.
## Dieses R-Skript basiert im Wesentlichen auf dem Skript Sensor_DHT22_V1.R und
## und wird nur um einen Einstieg in die Ereignisindentifizierung erweitert.
## Änderungen zur Version 1 werden mit V2 gekennzeichnet.
## Dieses Skript ist Bestandteil des Video "IoT und R, Daten versenden mit MQTT"
## und in diesem Video spielt das Datenversenden über MQTT eine besondere und
## die Datenauswertung über R eine untergeordnete Rolle.
## 
##
## Ad-Oculos-Projekt, https://www.faes.de/ad-oculos/
## Günter Faes
## Version 1.0, 15.08.2023
## R-Version: 4.3.1
## RStudio-Version: RStudio 2023.03.1+446 "Cherry Blossom"
##
################################################################################

## Pakete:
library(lubridate)    # Zeit-Berechnungen

################################################################################
## Beispieldaten:
# Diesen Daten wurden über das MicroPython-Programm Sensor_DHT22_P1.py gesammelt.

SensorDaten <- read.csv("DHT22.csv")
#SensorDaten <- read.csv("Log_1_2023-06-26.csv") # Zum Vergleichen der Ausführungsgeschw.

# V2:
# Aufgrund der Datenübertragung Pico -> MQTT -> Raspberry Pi fehlt die Spalten-
# überschrift im DataFrame. Diese wird hier zugefügt:
colnames(SensorDaten) <- c("Zeitstempel", "Temperatur", "Luftfeuchtigkeit")

################################################################################
## Datenübersicht:

View(SensorDaten)      # Tabellarische Übersicht

## Kenndaten über alle Beobachtungen:
n_Beobachtungen <- length(SensorDaten[,1])
# Kenndaten über die Zeit:
Messfrequenz <- dmy_hms(SensorDaten[3,1]) - dmy_hms(SensorDaten[2,1])
Messdauer_h <- difftime(dmy_hms(SensorDaten[n_Beobachtungen,1]), dmy_hms(SensorDaten[1,1]), units = "hours")
cat("Anzahl Messwerte: ", n_Beobachtungen)
cat("Messfrequenz: ", Messfrequenz, " Sekunden")
cat("Messdauer: ", round(Messdauer_h, digits = 1), " Stunden")

# Temperatur:
T_MW <- mean(SensorDaten[,2], na.rm = TRUE)       # Mittelwert
T_Stabw <- sd(SensorDaten[,2], na.rm = TRUE)      # Standardabweichung
T_min <- min(SensorDaten[,2])                     # Klienste Beobachtung
T_max <- max(SensorDaten[,2])                     # Größte Beobachtung
cat("Mittlere Temperatur über alle Beobachtungen: ", round(T_MW, digits = 1), " °C, Standardabweichung: ", round(T_Stabw, digits = 2), " °C", "\n")
cat("Tiefste gemessene Temperatur: ", T_min, " °C", "\n")
cat("Höchste gemessene Temperatur: ", T_max, " °C", "\n")

# Luftfeuchtigkeit:
Feuchte_MW <- mean(SensorDaten[,3], na.rm = TRUE)
Feuchte_Stabw <- sd(SensorDaten[,3], na.rm = TRUE)
Feuchte_min <- min(SensorDaten[,3])
Feuchte_max <- max(SensorDaten[,3])
cat("Mittlere Feuchtigkeit über alle Beobachtungen: ", round(Feuchte_MW, digits = 1), " %", ", Standardabweichung: ", round(Feuchte_Stabw, digits = 2), " %", "\n")
cat("Tiefste gemessene Luftfeuchtigkeit: ", Feuchte_min, " %", "\n")
cat("Höchste gemessene Luftfeuchtigkeit: ", Feuchte_max, " %", "\n")

# V2:
## Ereignisidentifizierung im Beobachtungszeitraum:
# Ein Ereignis in diesem Beispiel wird als "plötzlicher" Temperaturanstieg
# um einen Ereignisschwellwert ESW definiert. Es wird nur das erste Auftreten 
# eines Ereignisses gesucht.

Ereignis <- FALSE
E_Index <- 0
E_Zeitstempel <- ""
E_T <- 0.0
E_Feuchte <- 0.0
ESW <- 2.0 # 2,0 °C

for(i in 2:n_Beobachtungen)
{
  
    if(SensorDaten[i,2] >= (SensorDaten[i-1,2] + ESW))
      
    { # Ereignis eingetreten:
      Ereignis <- TRUE
      E_Index <- i
      E_Zeitstempel <- SensorDaten[i,1]
      E_T <- SensorDaten[i,2]
      E_Feuchte <- SensorDaten[i,3]
      
      break # Ereignis gefunden, for-Schleife verlassen
      
    }

}


## Grafische Darstellung:
par(mfrow = c(2,1))  # Die 2 Ausgabefenster vorbereiten

Zeitstempel_Start <- SensorDaten[1,1]
Zeitstempel_Ende <- SensorDaten[n_Beobachtungen,1]
X_Label <- paste("Index Messwert, Start:", Zeitstempel_Start, ", Ende:", Zeitstempel_Ende)
Subtitle <- paste("Messdauer ", round(Messdauer_h, digits = 1), "Stunden, ", "Messfrequenz: ", Messfrequenz, " Sekunden")

# Temperatur:
plot(SensorDaten$Temperatur,
     main = "Temperaturverlauf über den gesamten Messzeitraum",
     sub =  Subtitle,
     ylab = "Temperatur °C",
     xlab = X_Label,
     type = "p",
     col = "red")

# Hinzufügen der Mittwelwertlinie:
abline(h = T_MW, lty = 2, col = "red")

# V2:
# Ereignis einzeichnen:
if (Ereignis == TRUE)
{
  abline(v = E_Index, lty = 3, col = "red")
  
  xPos <- 0
  yPos <- 0
  
  if (E_Index >= 20)
    {
      xPos <- E_Index - 20
    }
    else
    {
      xPos <- 1
    }
  
  yPos <- T_max - 5
  
  text(xPos, yPos, "Ereignis bei:", pos = 3)
  text(xPos, yPos-2, E_Zeitstempel, pos = 3)
  text(xPos, yPos-4, paste(E_T, "°C"), pos = 3)
  text(xPos, yPos-6, paste("Index: ", E_Index), pos = 3)
  
}



# Luftfeuchtigkeit:
plot(SensorDaten$Luftfeuchtigkeit,
     main = "Luftfeuchtigkeitsverlauf über den gesamten Messzeitraum",
     sub =  Subtitle,
     ylab = "Luftfeuchtigkeit in %",
     xlab = X_Label,
     type = "p",
     col = "blue")

# Hinzufügen der Mittwelwertlinie:
abline(h = Feuchte_MW, lty = 2, col = "blue")


par(mfrow = c(1,1))   # Ausgangszustand Grafikfenster wieder herstellen


##### Skript-Ende #####

